<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\NewsletterCampaign;
use App\Models\Subscriber;
use App\Services\NewsletterService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\View\View;

class NewsletterCampaignController extends Controller
{
    public function index(): View
    {
        $campaigns = NewsletterCampaign::latest()->paginate(15);
        $subscriberStats = [
            'total' => Subscriber::count(),
            'active' => Subscriber::confirmed()->active()->count(),
        ];

        return view('admin.newsletters.index', compact('campaigns', 'subscriberStats'));
    }

    public function create(): View
    {
        $subscriberCount = Subscriber::confirmed()->active()->count();

        return view('admin.newsletters.create', compact('subscriberCount'));
    }

    public function store(Request $request, NewsletterService $newsletterService): RedirectResponse
    {
        $data = $request->validate([
            'subject' => ['required', 'string', 'max:255'],
            'content' => ['required', 'string'],
            'delivery_type' => ['required', 'in:now,schedule'],
            'scheduled_for' => ['nullable', 'date', 'after:now'],
        ]);

        if ($data['delivery_type'] === 'schedule' && empty($data['scheduled_for'])) {
            return back()->withInput()->withErrors([
                'scheduled_for' => 'Select when this newsletter should be delivered.',
            ]);
        }

        $campaign = NewsletterCampaign::create([
            'subject' => $data['subject'],
            'content' => $data['content'],
            'status' => 'draft',
            'created_by' => $request->user()?->id,
        ]);

        if ($data['delivery_type'] === 'now') {
            $sentCount = $newsletterService->sendCampaign($campaign);
            $message = $sentCount > 0
                ? "Newsletter sent to {$sentCount} subscribers."
                : 'Newsletter sent, but no confirmed subscribers were available.';
        } else {
            $scheduledAt = Carbon::parse($data['scheduled_for'], config('app.timezone'));
            $campaign->update([
                'status' => 'scheduled',
                'scheduled_at' => $scheduledAt,
            ]);
            $message = 'Newsletter scheduled for '.$scheduledAt->timezone(config('app.timezone'))->format('M j, Y g:i A');
        }

        return redirect()->route('admin.newsletters.index')->with('status', $message);
    }

    public function update(Request $request, NewsletterCampaign $newsletter, NewsletterService $newsletterService): RedirectResponse
    {
        $data = $request->validate([
            'action' => ['required', 'in:send_now,cancel,reschedule'],
            'scheduled_for' => ['nullable', 'date', 'after:now'],
        ]);

        $message = 'No changes made.';

        if ($data['action'] === 'send_now') {
            $sentCount = $newsletterService->sendCampaign($newsletter);
            $message = $sentCount > 0
                ? "Newsletter sent to {$sentCount} subscribers."
                : 'Newsletter sent, but no confirmed subscribers were available.';
        } elseif ($data['action'] === 'cancel') {
            $newsletter->markCancelled();
            $message = 'Newsletter schedule cancelled.';
        } elseif ($data['action'] === 'reschedule') {
            if (empty($data['scheduled_for'])) {
                return back()->withErrors([
                    'scheduled_for' => 'Pick a new schedule date.',
                ]);
            }

            $scheduledAt = Carbon::parse($data['scheduled_for'], config('app.timezone'));
            $newsletter->update([
                'status' => 'scheduled',
                'scheduled_at' => $scheduledAt,
                'sent_at' => null,
            ]);
            $message = 'Newsletter rescheduled for '.$scheduledAt->timezone(config('app.timezone'))->format('M j, Y g:i A');
        }

        return back()->with('status', $message);
    }

    public function destroy(NewsletterCampaign $newsletter): RedirectResponse
    {
        if ($newsletter->status === 'sent') {
            return back()->with('toast', 'Sent newsletters cannot be deleted.');
        }

        $newsletter->delete();

        return back()->with('status', 'Newsletter deleted.');
    }
}

