<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Post;
use Illuminate\Http\Request;

class PostController extends Controller
{
    public function index()
    {
        $this->authorize('viewAny', Post::class);
        $posts = Post::latest('published_at')->paginate(15);
        return view('admin.posts.index', compact('posts'));
    }

    public function create()
    {
        $this->authorize('create', Post::class);
        return view('admin.posts.create');
    }

    public function store(Request $request)
    {
        $this->authorize('create', Post::class);
        $data = $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255',
            'excerpt' => 'nullable|string',
            'body' => 'required|string',
            'is_published' => 'boolean',
            'published_at' => 'nullable|date',
        ]);
        $post = Post::create($data);
        \App\Services\AuditService::log('post.created', $post, $request->user());
        return redirect()->route('admin.posts.index')->with('status','Post created');
    }

    public function edit(Post $post)
    {
        $this->authorize('update', $post);
        return view('admin.posts.edit', compact('post'));
    }

    public function update(Request $request, Post $post)
    {
        $this->authorize('update', $post);
        $data = $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255',
            'excerpt' => 'nullable|string',
            'body' => 'required|string',
            'is_published' => 'boolean',
            'published_at' => 'nullable|date',
        ]);
        $post->update($data);
        \App\Services\AuditService::log('post.updated', $post, $request->user());
        return redirect()->route('admin.posts.index')->with('status','Post updated');
    }

    public function destroy(Post $post, Request $request)
    {
        $this->authorize('delete', $post);
        $post->delete();
        \App\Services\AuditService::log('post.deleted', $post, $request->user());
        return back()->with('status','Post deleted');
    }
}

