<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use App\Support\SiteSettings;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Storage;

class SiteController extends Controller
{
    public function edit(SiteSettings $siteSettings)
    {
        return view('admin.settings.site', [
            'appearance' => $siteSettings->appearance(),
            'typography' => $siteSettings->typography(),
            'content' => $siteSettings->content(),
            'brand' => $siteSettings->brand(),
        ]);
    }

    public function update(Request $request)
    {
        $rules = $this->validationRules();
        $data = $request->validate($rules);

        $this->persistGroup('appearance', $data['appearance'] ?? []);
        $this->persistGroup('typography', $data['typography'] ?? []);
        $this->persistContent($data['content'] ?? []);
        $this->persistBrand($request);

        Setting::flushAllCache();

        return back()->with('status', 'Site settings updated.');
    }

    protected function validationRules(): array
    {
        $colorRule = ['nullable', 'regex:/^#([0-9a-fA-F]{3}|[0-9a-fA-F]{6})$/'];

        $rules = [
            'appearance' => ['array'],
            'appearance.primary_color' => $colorRule,
            'appearance.primary_alt_color' => $colorRule,
            'appearance.accent_color' => $colorRule,
            'appearance.background_color' => $colorRule,
            'appearance.surface_color' => $colorRule,
            'appearance.text_color' => $colorRule,
            'appearance.muted_text_color' => $colorRule,
            'appearance.line_color' => $colorRule,
            'typography' => ['array'],
            'typography.heading_font' => ['nullable', 'string', 'max:120'],
            'typography.body_font' => ['nullable', 'string', 'max:120'],
            'typography.font_embed_url' => ['nullable', 'string', 'max:500'],
            'brand' => ['array'],
            'brand.logo' => ['nullable', 'image', 'max:2048'],
            'brand.favicon' => ['nullable', 'image', 'max:1024'],
            'brand.logo_remove' => ['nullable', 'boolean'],
            'brand.favicon_remove' => ['nullable', 'boolean'],
            'brand.home_hero' => ['nullable', 'image', 'max:4096'],
            'brand.home_hero_remove' => ['nullable', 'boolean'],
            'content' => ['array'],
        ];

        $contentDefaults = config('site.content', []);
        $this->addContentRules($rules, $contentDefaults, 'content');

        return $rules;
    }

    protected function addContentRules(array &$rules, array $defaults, string $baseKey): void
    {
        foreach ($defaults as $key => $value) {
            $ruleKey = $baseKey . '.' . $key;
            if (is_array($value)) {
                if (Arr::isAssoc($value)) {
                    $rules[$ruleKey] = ['array'];
                    $this->addContentRules($rules, $value, $ruleKey);
                } else {
                    $rules[$ruleKey] = ['nullable', 'string'];
                }
            } else {
                $rules[$ruleKey] = ['nullable', 'string'];
            }
        }
    }

    protected function persistGroup(string $group, array $values): void
    {
        foreach ($values as $key => $value) {
            $this->storeValue("site.$group.$key", $value);
        }
    }

    protected function persistContent(array $content): void
    {
        foreach ($content as $key => $value) {
            if (is_array($value)) {
                $this->persistContentRecursive("site.content.$key", $value);
            } else {
                $this->storeValue("site.content.$key", $value);
            }
        }
    }

    protected function persistContentRecursive(string $baseKey, array $values): void
    {
        foreach ($values as $key => $value) {
            if (is_array($value)) {
                $this->persistContentRecursive("$baseKey.$key", $value);
            } else {
                $this->storeValue("$baseKey.$key", $value);
            }
        }
    }

    protected function persistBrand(Request $request): void
    {
        if ($request->boolean('brand.logo_remove')) {
            $this->deleteStoredFile('site.brand.logo_path');
        }

        if ($request->boolean('brand.favicon_remove')) {
            $this->deleteStoredFile('site.brand.favicon_path');
        }

        if ($request->hasFile('brand.logo')) {
            $this->replaceStoredFile('site.brand.logo_path', $request->file('brand.logo')->store('site', 'public'));
        }

        if ($request->hasFile('brand.favicon')) {
            $this->replaceStoredFile('site.brand.favicon_path', $request->file('brand.favicon')->store('site', 'public'));
        }

        if ($request->boolean('brand.home_hero_remove')) {
            $this->deleteStoredFile('site.brand.home_hero_path');
        }

        if ($request->hasFile('brand.home_hero')) {
            $this->replaceStoredFile('site.brand.home_hero_path', $request->file('brand.home_hero')->store('site', 'public'));
        }
    }

    protected function deleteStoredFile(string $key): void
    {
        $current = Setting::get($key);
        if ($current) {
            Storage::disk('public')->delete($current);
            Setting::query()->where('key', $key)->delete();
        }
    }

    protected function replaceStoredFile(string $key, string $path): void
    {
        $current = Setting::get($key);
        if ($current && $current !== $path) {
            Storage::disk('public')->delete($current);
        }

        Setting::set($key, $path);
    }

    protected function storeValue(string $key, $value): void
    {
        $parsed = is_string($value) ? trim($value) : $value;

        if ($parsed === '' || $parsed === null) {
            Setting::query()->where('key', $key)->delete();
            return;
        }

        Setting::set($key, $parsed);
    }
}
