<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Laravel\Socialite\Facades\Socialite;

class SocialController extends Controller
{
    /**
     * Supported Socialite providers.
     * @var string[]
     */
    private array $providers = ['google', 'facebook', 'github', 'twitter', 'linkedin'];

    public function redirect(string $provider): RedirectResponse
    {
        abort_unless(in_array($provider, $this->providers, true), 404);
        return Socialite::driver($provider)->redirect();
    }

    public function callback(string $provider)
    {
        abort_unless(in_array($provider, $this->providers, true), 404);

        try {
            $socialUser = Socialite::driver($provider)->stateless()->user();
        } catch (\Throwable $e) {
            return redirect()->route('login')->withErrors(['social' => 'Could not authenticate with ' . Str::ucfirst($provider) . '.']);
        }

        $email = $socialUser->getEmail();
        // Some providers may not return email. If missing, deny and ask normal signup.
        if (!$email) {
            return redirect()->route('login')->withErrors(['social' => 'Your ' . Str::ucfirst($provider) . ' account has no email. Please use email/password.']);
        }

        $name = $socialUser->getName() ?: ($socialUser->getNickname() ?: Str::before($email, '@'));

        $user = User::firstOrCreate(
            ['email' => $email],
            [
                'name' => $name,
                'password' => Hash::make(Str::random(40)),
                'email_verified_at' => now(),
            ]
        );

        Auth::login($user, true);

        return redirect()->intended(route('dashboard'));
    }
}
