<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\Order;
use App\Models\Audit;
use App\Services\OrderFulfillmentService;

class WebhookController extends Controller
{
    public function stripe(Request $request)
    {
        $secret = config('services.stripe.webhook_secret');
        if (!$secret) {
            return response()->json(['error' => 'Webhook not configured'], 400);
        }

        $signature = $request->header('Stripe-Signature', '');
        $payload = $request->getContent();

        $timestamp = null; $v1 = null;
        foreach (explode(',', $signature) as $part) {
            [$k,$v] = array_pad(explode('=', trim($part), 2), 2, null);
            if ($k === 't') $timestamp = $v; elseif ($k === 'v1') $v1 = $v;
        }
        if (!$timestamp || !$v1) {
            return response()->json(['error' => 'Bad signature'], 400);
        }
        $signedPayload = $timestamp . '.' . $payload;
        $computed = hash_hmac('sha256', $signedPayload, $secret);
        if (!hash_equals($computed, $v1)) {
            return response()->json(['error' => 'Invalid signature'], 400);
        }

        $event = json_decode($payload, true) ?: [];
        $type = $event['type'] ?? '';
        $id = $event['id'] ?? null;

        if ($id && Audit::where('action', 'stripe.webhook')->where('meta->event_id', $id)->exists()) {
            return response()->json(['ok' => true]);
        }

        try {
            if (in_array($type, ['checkout.session.completed', 'payment_intent.succeeded'])) {
                $obj = $event['data']['object'] ?? [];
                $ref = $obj['metadata']['order_reference'] ?? null;
                $amount = (int) ($obj['amount_total'] ?? ($obj['amount'] ?? 0));
                $currency = strtoupper($obj['currency'] ?? '');

                if ($ref) {
                    $order = Order::where('reference', $ref)->first();
                    if ($order && $order->status !== 'paid') {
                        if ($amount > 0 && (int)$order->amount_kobo !== (int)$amount) {
                            \App\Services\AuditService::log('payment.mismatch', $order, null, [
                                'provider' => 'stripe', 'amount' => $amount, 'currency' => $currency,
                            ]);
                        } else {
                            $order->update(['status' => 'paid', 'placed_at' => now(), 'provider' => 'stripe']);
                            $order->refresh();
                            app(OrderFulfillmentService::class)->handlePaidOrder($order);
                        }
                    }
                }
            }

            \App\Services\AuditService::log('stripe.webhook', null, null, [
                'event_id' => $id, 'type' => $type,
            ]);
        } catch (\Throwable $e) {
            Log::error('Stripe webhook error: '.$e->getMessage(), ['event' => $event]);
            return response()->json(['error' => 'Unhandled'], 500);
        }

        return response()->json(['ok' => true]);
    }

    public function paypal(Request $request)
    {
        $base = rtrim(config('services.paypal.base_url', 'https://api-m.paypal.com'), '/');
        $authId = $request->header('paypal-auth-algo');
        $transId = $request->header('paypal-transmission-id');
        $certUrl = $request->header('paypal-cert-url');
        $transSig = $request->header('paypal-transmission-sig');
        $transTime = $request->header('paypal-transmission-time');
        $webhookId = config('services.paypal.webhook_id');
        $payload = $request->getContent();

        if (!$webhookId && !app()->isLocal()) {
            return response()->json(['error' => 'Webhook not configured'], 400);
        }

        $token = $this->paypalToken();
        if (!$token) {
            return response()->json(['error' => 'PayPal not configured'], 400);
        }

        $verifyRes = Http::withToken($token)->post("$base/v1/notifications/verify-webhook-signature", array_filter([
            'auth_algo' => $authId,
            'cert_url' => $certUrl,
            'transmission_id' => $transId,
            'transmission_sig' => $transSig,
            'transmission_time' => $transTime,
            'webhook_id' => $webhookId,
            'webhook_event' => json_decode($payload, true),
        ]));

        if ($verifyRes->failed() || strtoupper(($verifyRes->json()['verification_status'] ?? '')) !== 'SUCCESS') {
            return response()->json(['error' => 'Invalid signature'], 400);
        }

        $event = json_decode($payload, true) ?: [];
        $eventId = $event['id'] ?? null;
        $eventType = $event['event_type'] ?? '';

        if ($eventId && Audit::where('action', 'paypal.webhook')->where('meta->event_id', $eventId)->exists()) {
            return response()->json(['ok' => true]);
        }

        try {
            if (in_array($eventType, ['CHECKOUT.ORDER.APPROVED', 'PAYMENT.CAPTURE.COMPLETED'])) {
                $resource = $event['resource'] ?? [];
                $ref = $resource['purchase_units'][0]['custom_id'] ?? ($resource['supplementary_data']['related_ids']['order_id'] ?? null);
                $amount = null; $currency = null;
                if (!empty($resource['amount'])) {
                    $currency = strtoupper($resource['amount']['currency_code'] ?? '');
                    $amount = (int) round(((float) ($resource['amount']['value'] ?? 0)) * 100);
                } elseif (!empty($resource['purchase_units'][0]['amount'])) {
                    $unit = $resource['purchase_units'][0]['amount'];
                    $currency = strtoupper($unit['currency_code'] ?? '');
                    $amount = (int) round(((float) ($unit['value'] ?? 0)) * 100);
                }

                if ($ref) {
                    $order = Order::where('reference', $ref)->first();
                    if ($order && $order->status !== 'paid') {
                        if ($amount > 0 && (int)$order->amount_kobo !== (int)$amount) {
                            \App\Services\AuditService::log('payment.mismatch', $order, null, [
                                'provider' => 'paypal', 'amount' => $amount, 'currency' => $currency,
                            ]);
                        } else {
                            $order->update(['status' => 'paid', 'placed_at' => now(), 'provider' => 'paypal']);
                            $order->refresh();
                            app(OrderFulfillmentService::class)->handlePaidOrder($order);
                        }
                    }
                }
            }

            \App\Services\AuditService::log('paypal.webhook', null, null, [
                'event_id' => $eventId, 'type' => $eventType,
            ]);
        } catch (\Throwable $e) {
            Log::error('PayPal webhook error: '.$e->getMessage(), ['event' => $event]);
            return response()->json(['error' => 'Unhandled'], 500);
        }

        return response()->json(['ok' => true]);
    }

    private function paypalToken(): ?string
    {
        $clientId = config('services.paypal.client_id');
        $secret = config('services.paypal.secret');
        $base = rtrim(config('services.paypal.base_url', 'https://api-m.paypal.com'), '/');
        if (!$clientId || !$secret) return null;
        $res = Http::asForm()->withBasicAuth($clientId, $secret)->post("$base/v1/oauth2/token", ['grant_type' => 'client_credentials']);
        if ($res->failed()) return null;
        return $res->json()['access_token'] ?? null;
    }
}
