<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class Product extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'slug',
        'type',
        'excerpt',
        'description',
        'price_kobo',
        'sale_price_kobo',
        'shipping_fee_kobo',
        'stock',
        'cover_path',
        'download_path',
        'download_disk',
        'has_hardcopy',
        'hardcopy_price_kobo',
        'is_featured',
        'published_at',
        'category_id',
    ];

    protected $casts = [
        'is_featured' => 'boolean',
        'has_hardcopy' => 'boolean',
        'published_at' => 'datetime',
        'shipping_fee_kobo' => 'integer',
        'hardcopy_price_kobo' => 'integer',
    ];

    public function getPriceAttribute(): int
    {
        return (int) ($this->sale_price_kobo ?? $this->price_kobo);
    }

    public function getHardcopyPriceAttribute(): ?int
    {
        if (! $this->has_hardcopy) {
            return null;
        }

        return (int) ($this->hardcopy_price_kobo ?? $this->price_kobo);
    }

    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    public function getIsDigitalAttribute(): bool
    {
        if ($this->download_path) {
            return true;
        }

        return in_array($this->type, ['ebook', 'audiobook', 'digital', 'softcopy'], true);
    }

    // Scopes
    public function scopeDiscounted($query)
    {
        return $query->whereNotNull('sale_price_kobo')->whereColumn('sale_price_kobo', '<', 'price_kobo');
    }

    public function scopeLatestPublished($query)
    {
        return $query->latest('published_at');
    }

    protected static function booted(): void
    {
        static::creating(function ($model) {
            if (empty($model->slug)) {
                $model->slug = Str::slug($model->title) . '-' . Str::random(4);
            }
        });
    }

    public function getCoverUrlAttribute(): ?string
    {
        $path = $this->cover_path;

        if (!$path) {
            return null;
        }

        if (Str::startsWith($path, ['http://', 'https://', '//'])) {
            return $path;
        }

        $normalized = ltrim($path, '/');
        $relative = Str::startsWith($normalized, 'storage/')
            ? ltrim(Str::after($normalized, 'storage/'), '/')
            : $normalized;

        $disk = $this->cover_disk ?? 'public';
        $storage = Storage::disk($disk);

        if ($storage->exists($relative)) {
            return $storage->url($relative);
        }

        $publicPath = public_path('storage/' . $relative);
        if (file_exists($publicPath)) {
            return asset('storage/' . $relative);
        }

        return asset($normalized);
    }
}

