<?php

namespace App\Services;

use App\Models\Product;
use Illuminate\Support\Facades\Session;

class CartService
{
    protected string $key = 'cart.items';

    public function items(): array
    {
        $items = Session::get($this->key, []);
        $updated = false;

        foreach ($items as $key => $item) {
            $productId = (int) ($item['product_id'] ?? 0);
            $variant = $item['variant'] ?? 'digital';
            $product = Product::find($productId);

            if ($product) {
                $normalized = $this->buildLineItem($product, $variant, (int) ($item['qty'] ?? 0));
                $items[$key] = array_merge($item, $normalized);
            } else {
                $items[$key]['shipping_fee_kobo'] = (int) ($item['shipping_fee_kobo'] ?? 0);
                $items[$key]['price_kobo'] = (int) ($item['price_kobo'] ?? 0);
                $items[$key]['qty'] = (int) ($item['qty'] ?? 0);
            }
        }

        if ($updated) {
            Session::put($this->key, $items);
        }

        return $items;
    }

    public function add(Product $product, string $variant = 'digital', int $qty = 1): void
    {
        $variant = $this->normalizeVariant($variant, $product);
        $isDigital = $variant === 'digital';
        if ($isDigital) {
            $qty = 1;
        }
        $key = $this->keyFor($product->id, $variant);
        $items = Session::get($this->key, []);

        if (!$isDigital && isset($items[$key])) {
            $qty += (int) ($items[$key]['qty'] ?? 0);
        }

        $items[$key] = $this->buildLineItem($product, $variant, $qty);
        Session::put($this->key, $items);
    }

    public function update(int $productId, string $variant, int $qty): void
    {
        $variant = $this->normalizeVariant($variant);
        if ($variant === 'digital') {
            $qty = 1;
        }
        $key = $this->keyFor($productId, $variant);
        $items = Session::get($this->key, []);
        if (!isset($items[$key])) {
            return;
        }
        if ($qty <= 0) {
            unset($items[$key]);
        } else {
            $items[$key]['qty'] = $qty;
        }
        Session::put($this->key, $items);
    }

    public function remove(int $productId, string $variant): void
    {
        $variant = $this->normalizeVariant($variant);
        $items = Session::get($this->key, []);
        unset($items[$this->keyFor($productId, $variant)]);
        Session::put($this->key, $items);
    }

    public function clear(): void
    {
        Session::forget($this->key);
    }

    public function quantity(): int
    {
        return array_sum(array_map(fn ($i) => (int) $i['qty'], $this->items()));
    }

    public function totalKobo(): int
    {
        return array_sum(array_map(function ($item) {
            $price = (int) ($item['price_kobo'] ?? 0);
            $shipping = (int) ($item['shipping_fee_kobo'] ?? 0);
            $qty = (int) ($item['qty'] ?? 0);
            return ($price + $shipping) * $qty;
        }, $this->items()));
    }

    public function shippingKobo(): int
    {
        return array_sum(array_map(function ($item) {
            return (int) ($item['shipping_fee_kobo'] ?? 0) * (int) ($item['qty'] ?? 0);
        }, $this->items()));
    }

    public function toOrderPayload(): array
    {
        $items = $this->items();
        return [
            'items' => array_values($items),
            'amount_kobo' => $this->totalKobo(),
            'shipping_kobo' => $this->shippingKobo(),
        ];
    }

    protected function buildLineItem(Product $product, string $variant, int $qty): array
    {
        $variant = $this->normalizeVariant($variant, $product);
        $isDigital = $variant === 'digital';
        $price = $isDigital ? $product->price : ($product->hardcopy_price ?? $product->price_kobo);
        $shipping = $isDigital ? 0 : (int) $product->shipping_fee_kobo;

        return [
            'key' => $this->keyFor($product->id, $variant),
            'product_id' => $product->id,
            'variant' => $variant,
            'variant_label' => $isDigital ? strtoupper($product->type) : 'Hardcopy',
            'title' => $product->title,
            'price_kobo' => (int) $price,
            'shipping_fee_kobo' => $shipping,
            'product_type' => $product->type,
            'is_digital' => $isDigital,
            'qty' => $isDigital ? 1 : max(1, $qty),
            'cover_path' => $product->cover_path,
            'cover_url' => $product->cover_url,
            'slug' => $product->slug,
        ];
    }

    protected function normalizeVariant(?string $variant, ?Product $product = null): string
    {
        $variant = strtolower($variant ?? 'digital');
        if ($variant === 'hardcopy') {
            return 'hardcopy';
        }
        return 'digital';
    }

    protected function keyFor(int $productId, string $variant): string
    {
        return $productId . ':' . $variant;
    }
}
