﻿<?php

namespace App\Services\Payments;

use App\Models\Setting;
use Illuminate\Support\Facades\Http;

class StripeGateway
{
    public function init(int $amountCents, string $email, array $metadata): array
    {
        $base = rtrim(Setting::get('stripe.base_url', config('services.stripe.base_url', 'https://api.stripe.com')), '/');
        $secret = Setting::get('stripe.secret', config('services.stripe.secret'));
        $reference = $metadata['reference'] ?? ('order_'.uniqid());
        $successUrl = $metadata['success_url'] ?? route('checkout.verify').'?provider=stripe&reference='.urlencode($reference).'&session_id={CHECKOUT_SESSION_ID}';
        $cancelUrl = $metadata['cancel_url'] ?? route('cart.index');

        $payload = [
            'mode' => 'payment',
            'success_url' => $successUrl,
            'cancel_url' => $cancelUrl,
            'customer_email' => $email,
            'metadata' => array_filter([
                'order_id' => $metadata['order_id'] ?? null,
                'order_reference' => $reference,
            ]),
            'line_items' => [[
                'quantity' => 1,
                'price_data' => [
                    'currency' => strtolower(config('payment.currency', 'USD')),
                    'product_data' => [
                        'name' => $metadata['description'] ?? ('Order '.$reference),
                    ],
                    'unit_amount' => $amountCents,
                ],
            ]],
        ];

        if ($secret) {
            $response = Http::asForm()->withToken($secret)->post("{$base}/v1/checkout/sessions", $this->formify($payload));
            $json = $response->json();
            if (!empty($json['url'])) {
                return [
                    'checkout_url' => $json['url'],
                    'session_id' => $json['id'] ?? null,
                    'reference' => $reference,
                ];
            }
        }

        $sessionId = $metadata['fallback_session'] ?? ('cs_test_'.uniqid());
        return [
            'checkout_url' => route('checkout.verify').'?provider=stripe&reference='.urlencode($reference).'&session_id='.urlencode($sessionId).'&dev=1',
            'session_id' => $sessionId,
            'reference' => $reference,
        ];
    }

    public function verify(string $sessionId): array
    {
        $base = rtrim(Setting::get('stripe.base_url', config('services.stripe.base_url', 'https://api.stripe.com')), '/');
        $secret = Setting::get('stripe.secret', config('services.stripe.secret'));

        if ($secret && $sessionId) {
            $response = Http::withToken($secret)->get("{$base}/v1/checkout/sessions/{$sessionId}", ['expand' => ['payment_intent']]);
            $json = $response->json();
            $status = strtolower($json['status'] ?? '');
            $amount = (int) ($json['amount_total'] ?? 0);
            return [
                'status' => $status === 'complete' ? 'success' : 'failed',
                'amount_cents' => $amount,
                'currency' => strtoupper($json['currency'] ?? config('payment.currency', 'USD')),
                'metadata' => $json['metadata'] ?? [],
                'reference' => $json['metadata']['order_reference'] ?? null,
            ];
        }

        return [
            'status' => 'success',
            'amount_cents' => 0,
            'currency' => config('payment.currency', 'USD'),
            'metadata' => [],
            'reference' => null,
        ];
    }

    private function formify(array $data, string $prefix = ''): array
    {
        $form = [];
        foreach ($data as $key => $value) {
            $compoundKey = $prefix === '' ? (string) $key : $prefix.'['.$key.']';
            if (is_array($value)) {
                $form += $this->formify($value, $compoundKey);
            } elseif (!is_null($value)) {
                $form[$compoundKey] = $value;
            }
        }
        return $form;
    }
}

