<?php

namespace App\Support;

use Illuminate\Support\Collection;

class CountryRepository
{
    protected static ?Collection $dataset = null;
    protected static ?Collection $countries = null;
    protected static array $subdivisions = [];
    protected static array $countryLookup = [];
    protected static array $manualSubdivisions = [
        'US' => [
            ['code' => 'AL', 'name' => 'Alabama'],
            ['code' => 'AK', 'name' => 'Alaska'],
            ['code' => 'AZ', 'name' => 'Arizona'],
            ['code' => 'AR', 'name' => 'Arkansas'],
            ['code' => 'CA', 'name' => 'California'],
            ['code' => 'CO', 'name' => 'Colorado'],
            ['code' => 'CT', 'name' => 'Connecticut'],
            ['code' => 'DE', 'name' => 'Delaware'],
            ['code' => 'DC', 'name' => 'District of Columbia'],
            ['code' => 'FL', 'name' => 'Florida'],
            ['code' => 'GA', 'name' => 'Georgia'],
            ['code' => 'HI', 'name' => 'Hawaii'],
            ['code' => 'ID', 'name' => 'Idaho'],
            ['code' => 'IL', 'name' => 'Illinois'],
            ['code' => 'IN', 'name' => 'Indiana'],
            ['code' => 'IA', 'name' => 'Iowa'],
            ['code' => 'KS', 'name' => 'Kansas'],
            ['code' => 'KY', 'name' => 'Kentucky'],
            ['code' => 'LA', 'name' => 'Louisiana'],
            ['code' => 'ME', 'name' => 'Maine'],
            ['code' => 'MD', 'name' => 'Maryland'],
            ['code' => 'MA', 'name' => 'Massachusetts'],
            ['code' => 'MI', 'name' => 'Michigan'],
            ['code' => 'MN', 'name' => 'Minnesota'],
            ['code' => 'MS', 'name' => 'Mississippi'],
            ['code' => 'MO', 'name' => 'Missouri'],
            ['code' => 'MT', 'name' => 'Montana'],
            ['code' => 'NE', 'name' => 'Nebraska'],
            ['code' => 'NV', 'name' => 'Nevada'],
            ['code' => 'NH', 'name' => 'New Hampshire'],
            ['code' => 'NJ', 'name' => 'New Jersey'],
            ['code' => 'NM', 'name' => 'New Mexico'],
            ['code' => 'NY', 'name' => 'New York'],
            ['code' => 'NC', 'name' => 'North Carolina'],
            ['code' => 'ND', 'name' => 'North Dakota'],
            ['code' => 'OH', 'name' => 'Ohio'],
            ['code' => 'OK', 'name' => 'Oklahoma'],
            ['code' => 'OR', 'name' => 'Oregon'],
            ['code' => 'PA', 'name' => 'Pennsylvania'],
            ['code' => 'RI', 'name' => 'Rhode Island'],
            ['code' => 'SC', 'name' => 'South Carolina'],
            ['code' => 'SD', 'name' => 'South Dakota'],
            ['code' => 'TN', 'name' => 'Tennessee'],
            ['code' => 'TX', 'name' => 'Texas'],
            ['code' => 'UT', 'name' => 'Utah'],
            ['code' => 'VT', 'name' => 'Vermont'],
            ['code' => 'VA', 'name' => 'Virginia'],
            ['code' => 'WA', 'name' => 'Washington'],
            ['code' => 'WV', 'name' => 'West Virginia'],
            ['code' => 'WI', 'name' => 'Wisconsin'],
            ['code' => 'WY', 'name' => 'Wyoming'],
        ],
        'CA' => [
            ['code' => 'AB', 'name' => 'Alberta'],
            ['code' => 'BC', 'name' => 'British Columbia'],
            ['code' => 'MB', 'name' => 'Manitoba'],
            ['code' => 'NB', 'name' => 'New Brunswick'],
            ['code' => 'NL', 'name' => 'Newfoundland and Labrador'],
            ['code' => 'NS', 'name' => 'Nova Scotia'],
            ['code' => 'NT', 'name' => 'Northwest Territories'],
            ['code' => 'NU', 'name' => 'Nunavut'],
            ['code' => 'ON', 'name' => 'Ontario'],
            ['code' => 'PE', 'name' => 'Prince Edward Island'],
            ['code' => 'QC', 'name' => 'Quebec'],
            ['code' => 'SK', 'name' => 'Saskatchewan'],
            ['code' => 'YT', 'name' => 'Yukon'],
        ],
    ];

    public static function countries(): Collection
    {
        if (! static::$countries instanceof Collection) {
            static::$countries = static::dataset()
                ->map(function (array $country) {
                    return [
                        'code' => strtoupper($country['code'] ?? ''),
                        'name' => $country['name'] ?? '',
                    ];
                })
                ->filter(fn ($country) => ! empty($country['code']) && ! empty($country['name']))
                ->unique('code')
                ->sortBy('name')
                ->values();
        }

        return static::$countries;
    }

    public static function subdivisions(string $countryCode): Collection
    {
        $countryCode = strtoupper(trim($countryCode));

        if (isset(static::$subdivisions[$countryCode])) {
            return static::$subdivisions[$countryCode];
        }

        $country = static::countryData($countryCode);

        if (! $country || empty($country['states']) || ! is_array($country['states'])) {
            if (isset(static::$manualSubdivisions[$countryCode])) {
                return static::$subdivisions[$countryCode] = collect(static::$manualSubdivisions[$countryCode]);
            }
            return static::$subdivisions[$countryCode] = collect();
        }

        $items = collect($country['states'])
            ->filter(fn ($subdivision) => ! empty($subdivision['code']) || ! empty($subdivision['name']))
            ->map(function (array $subdivision) {
                $name = $subdivision['name'] ?? $subdivision['code'] ?? 'Unknown';
                $code = $subdivision['code'] ?? $name;

                return [
                    'code' => strtoupper($code),
                    'name' => $name,
                ];
            })
            ->sortBy('name')
            ->values();

        return static::$subdivisions[$countryCode] = $items;
    }

    protected static function dataset(): Collection
    {
        if (static::$dataset instanceof Collection) {
            return static::$dataset;
        }

        $path = resource_path('data/countries_states.json');

        if (! is_file($path)) {
            return static::$dataset = collect();
        }

        try {
            $data = json_decode(file_get_contents($path), true, flags: JSON_THROW_ON_ERROR);
        } catch (\Throwable) {
            $data = [];
        }

        $collection = collect($data);
        static::$countryLookup = $collection
            ->filter(fn ($country) => ! empty($country['code']))
            ->mapWithKeys(fn ($country) => [strtoupper($country['code']) => $country])
            ->all();

        return static::$dataset = $collection;
    }

    protected static function countryData(string $countryCode): ?array
    {
        $countryCode = strtoupper($countryCode);

        static::dataset(); // ensure lookup is populated

        return static::$countryLookup[$countryCode] ?? null;
    }
}
