@extends('layouts.admin')

@section('content')
@php
    $priceAmount = old('price_amount');
    if ($priceAmount === null) {
        $priceAmount = number_format($product->price_kobo / 100, 2, '.', '');
    }

    $salePriceAmount = old('sale_price_amount');
    if ($salePriceAmount === null) {
        $salePriceAmount = $product->sale_price_kobo !== null
            ? number_format($product->sale_price_kobo / 100, 2, '.', '')
            : '';
    }
    $hasSalePrice = $salePriceAmount !== '' && $salePriceAmount !== null;

    $hasHardcopyInput = old('has_hardcopy');
    $hasHardcopy = $hasHardcopyInput !== null ? (bool) $hasHardcopyInput : (bool) $product->has_hardcopy;

    $hardcopyPriceAmount = old('hardcopy_price_amount');
    if ($hardcopyPriceAmount === null) {
        $hardcopyPriceAmount = $product->hardcopy_price_kobo !== null
            ? number_format($product->hardcopy_price_kobo / 100, 2, '.', '')
            : '';
    }

    $stockValue = old('stock');
    if ($stockValue === null) {
        $stockValue = $product->stock;
    }

    $shippingFeeAmount = old('shipping_fee_amount');
    if ($shippingFeeAmount === null) {
        $shippingFeeAmount = $hasHardcopy
            ? number_format(($product->shipping_fee_kobo ?? 0) / 100, 2, '.', '')
            : '';
    }

    $publishedAt = old('published_at');
    if ($publishedAt === null && $product->published_at) {
        $publishedAt = $product->published_at->format('Y-m-d\TH:i');
    }

    $isFeatured = old('is_featured') !== null
        ? (bool) old('is_featured')
        : (bool) $product->is_featured;

    $rawCoverPath = (string) ($product->cover_path ?? '');
    $coverDisk = $product->cover_disk ?? 'public';
    $isRemoteCover = $rawCoverPath !== '' && \Illuminate\Support\Str::startsWith($rawCoverPath, ['http://', 'https://', '//']);
    $normalizedCoverPath = $isRemoteCover ? '' : ltrim($rawCoverPath, '/');
    $relativeCoverPath = $normalizedCoverPath !== ''
        ? ltrim(\Illuminate\Support\Str::startsWith($normalizedCoverPath, 'storage/') ? \Illuminate\Support\Str::after($normalizedCoverPath, 'storage/') : $normalizedCoverPath, '/')
        : '';
    $coverExistsOnDisk = $relativeCoverPath !== '' && \Illuminate\Support\Facades\Storage::disk($coverDisk)->exists($relativeCoverPath);
    $coverExistsInPublic = $relativeCoverPath !== '' && file_exists(public_path('storage/'.$relativeCoverPath));
    $coverUrl = $isRemoteCover
        ? $rawCoverPath
        : ($coverExistsOnDisk
            ? \Illuminate\Support\Facades\Storage::disk($coverDisk)->url($relativeCoverPath)
            : ($coverExistsInPublic ? asset('storage/'.$relativeCoverPath) : $product->cover_url));
@endphp

<div class="max-w-3xl mx-auto px-4 py-8" data-aos="fade-up">
    <h1 class="heading text-2xl mb-4">Edit Product</h1>

    @if (session('status'))
        <div class="mb-3 rounded border border-emerald-300 bg-emerald-50 px-4 py-2 text-emerald-800">
            {{ session('status') }}
        </div>
    @endif

    @if ($errors->any())
        <div class="mb-3 rounded border border-red-300 bg-red-50 px-4 py-2 text-red-800">
            <div class="font-semibold mb-1">Please fix the following:</div>
            <ul class="list-disc ms-5">
                @foreach ($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
        </div>
    @endif

    <form action="{{ route('admin.products.update', $product) }}" method="post" enctype="multipart/form-data" class="space-y-3 bg-white border border-slate-200 rounded p-4">
        @csrf
        @method('PUT')

        <input
            name="title"
            class="w-full border border-slate-300 rounded px-3 py-2"
            placeholder="Title"
            value="{{ old('title', $product->title) }}"
            required
        >

        <div class="grid grid-cols-1 gap-3 md:grid-cols-2">
            @php($types = ['ebook','audiobook','magazine','journal','comic'])
            <select name="type" class="border border-slate-300 rounded px-3 py-2" required>
                @foreach($types as $type)
                    <option value="{{ $type }}" @selected(old('type', $product->type) === $type)>{{ ucfirst($type) }}</option>
                @endforeach
                @if($product->type && ! in_array($product->type, $types, true))
                    <option value="{{ $product->type }}" selected>{{ ucfirst($product->type) }}</option>
                @endif
            </select>

            <div>
                <input type="file" name="cover_image" accept="image/*" class="border border-slate-300 rounded px-3 py-2 bg-white w-full">
                <div id="dropzone-edit" class="mt-3 border-2 border-dashed border-slate-300 rounded p-4 text-center text-slate-500">
                    Drag and drop cover image here or click the file input above.
                    <div class="mt-3">
                        <img
                            id="preview-edit"
                            src="{{ $coverUrl }}"
                            data-existing-url="{{ $coverUrl }}"
                            class="mx-auto h-32 rounded border border-slate-200 {{ $coverUrl ? '' : 'hidden' }}"
                            alt="Current cover preview"
                        >
                    </div>
                </div>
                @if($rawCoverPath !== '' && ! $isRemoteCover && ! $coverExistsOnDisk && ! $coverExistsInPublic)
                    <p class="mt-2 text-xs text-amber-600">The stored cover reference points to "{{ $rawCoverPath }}", but the file is missing. Upload a new cover to replace it.</p>
                @endif
            </div>
        </div>

        <div class="grid grid-cols-1 gap-3 md:grid-cols-2">
            <select name="category_id" class="border border-slate-300 rounded px-3 py-2">
                <option value="">-- Select category --</option>
                @foreach(\App\Models\Category::orderBy('name')->get() as $category)
                    <option value="{{ $category->id }}" @selected((int) old('category_id', $product->category_id) === $category->id)>{{ $category->name }}</option>
                @endforeach
            </select>
            <input
                name="category_name"
                class="border border-slate-300 rounded px-3 py-2"
                placeholder="Or create new category"
                value="{{ old('category_name') }}"
            >
        </div>

        <div class="grid grid-cols-1 gap-3 md:grid-cols-2 items-center">
            <input
                type="number"
                step="0.01"
                name="price_amount"
                class="border border-slate-300 rounded px-3 py-2"
                placeholder="Digital price (USD)"
                value="{{ $priceAmount }}"
                required
            >
            <div class="flex items-center gap-2">
                <label class="inline-flex items-center gap-2">
                    <input type="checkbox" id="on_sale_toggle_edit" @checked($hasSalePrice)>
                    <span>On sale</span>
                </label>
                <input
                    type="number"
                    step="0.01"
                    name="sale_price_amount"
                    id="sale_price_input_edit"
                    class="border border-slate-300 rounded px-3 py-2 {{ $hasSalePrice ? '' : 'hidden' }}"
                    placeholder="Sale price (USD)"
                    value="{{ $salePriceAmount }}"
                    @required($hasSalePrice)
                >
            </div>
        </div>

        <div class="grid grid-cols-1 gap-3 md:grid-cols-2 items-center">
            <label class="flex items-center gap-2">
                <input type="checkbox" name="has_hardcopy" value="1" id="hardcopy_toggle_edit" @checked($hasHardcopy)>
                <span>Hardcopy available</span>
            </label>
            <input
                type="number"
                step="0.01"
                min="0"
                name="hardcopy_price_amount"
                id="hardcopy_price_input_edit"
                class="border border-slate-300 rounded px-3 py-2 {{ $hasHardcopy ? '' : 'hidden' }}"
                placeholder="Hardcopy price (USD)"
                value="{{ $hardcopyPriceAmount }}"
                @disabled(! $hasHardcopy)
                @required($hasHardcopy)
            >
        </div>

        <div class="grid grid-cols-1 gap-3 md:grid-cols-2 {{ $hasHardcopy ? '' : 'hidden' }}" id="hardcopy_inventory_fields_edit">
            <input
                type="number"
                name="stock"
                id="stock_input_edit"
                class="border border-slate-300 rounded px-3 py-2"
                placeholder="Hardcopy stock"
                min="0"
                value="{{ $stockValue }}"
                @disabled(! $hasHardcopy)
                @required($hasHardcopy)
            >
            <input
                type="number"
                step="0.01"
                min="0"
                name="shipping_fee_amount"
                id="shipping_fee_input_edit"
                class="border border-slate-300 rounded px-3 py-2"
                placeholder="Hardcopy shipping fee (USD)"
                value="{{ $shippingFeeAmount }}"
                @disabled(! $hasHardcopy)
            >
        </div>

        <div class="grid grid-cols-1 gap-3">
            <div>
                <input
                    type="file"
                    name="download_file"
                    class="border border-slate-300 rounded px-3 py-2 bg-white w-full"
                    accept=".pdf,.epub,.mobi,.zip,.mp3,.wav,.mp4,.doc,.docx,.txt,.rtf,.ppt,.pptx,.xls,.xlsx,.aiff,.flac"
                >
                @if($product->download_path)
                    <p class="text-xs text-slate-500 mt-1">Current file: {{ basename($product->download_path) }}</p>
                    <label class="inline-flex items-center gap-2 text-xs text-slate-600 mt-1">
                        <input type="checkbox" name="remove_download" value="1" @checked(old('remove_download'))> Remove existing file
                    </label>
                @endif
            </div>
        </div>

        <p class="text-xs text-slate-500">Maintain the digital details above. If the hardcopy option is enabled, remember to provide pricing (and optional shipping fee) for the physical edition.</p>

        <textarea
            name="excerpt"
            rows="3"
            class="w-full border border-slate-300 rounded px-3 py-2"
            placeholder="Short excerpt"
        >{{ old('excerpt', $product->excerpt) }}</textarea>

        <textarea
            name="description"
            rows="6"
            class="w-full border border-slate-300 rounded px-3 py-2"
            placeholder="Description"
        >{{ old('description', $product->description) }}</textarea>

        <div class="grid grid-cols-1 gap-3 md:grid-cols-2">
            <label class="flex items-center gap-2">
                <input type="checkbox" name="is_featured" value="1" @checked($isFeatured)>
                <span>Featured</span>
            </label>
            <input
                type="datetime-local"
                name="published_at"
                class="border border-slate-300 rounded px-3 py-2"
                value="{{ $publishedAt }}"
            >
        </div>

        <button class="btn">Save</button>
    </form>
</div>

<script>
document.addEventListener('DOMContentLoaded', () => {
    const fileInput = document.querySelector('input[name="cover_image"]');
    const dropzone = document.getElementById('dropzone-edit');
    const preview = document.getElementById('preview-edit');
    const existingUrl = preview?.dataset.existingUrl;

    const showPreview = (file) => {
        const reader = new FileReader();
        reader.onload = (event) => {
            if (!preview) return;
            preview.src = event.target.result;
            preview.classList.remove('hidden');
        };
        reader.readAsDataURL(file);
    };

    if (existingUrl && preview) {
        preview.src = existingUrl;
        preview.classList.remove('hidden');
    }

    fileInput?.addEventListener('change', (event) => {
        const file = event.target.files?.[0];
        if (file) {
            showPreview(file);
        }
    });

    dropzone?.addEventListener('dragover', (event) => {
        event.preventDefault();
        dropzone.classList.add('bg-slate-50');
    });

    dropzone?.addEventListener('dragleave', () => {
        dropzone.classList.remove('bg-slate-50');
    });

    dropzone?.addEventListener('drop', (event) => {
        event.preventDefault();
        dropzone.classList.remove('bg-slate-50');
        const file = event.dataTransfer.files?.[0];
        if (!file) return;
        const dataTransfer = new DataTransfer();
        dataTransfer.items.add(file);
        if (fileInput) {
            fileInput.files = dataTransfer.files;
        }
        showPreview(file);
    });

    const saleToggle = document.getElementById('on_sale_toggle_edit');
    const saleInput = document.getElementById('sale_price_input_edit');
    const syncSale = () => {
        if (!saleInput) return;
        if (saleToggle?.checked) {
            saleInput.classList.remove('hidden');
            saleInput.required = true;
        } else {
            saleInput.classList.add('hidden');
            saleInput.required = false;
            saleInput.value = '';
        }
    };
    saleToggle?.addEventListener('change', syncSale);
    syncSale();

    const hardcopyToggle = document.getElementById('hardcopy_toggle_edit');
    const hardcopyPrice = document.getElementById('hardcopy_price_input_edit');
    const hardcopyFields = document.getElementById('hardcopy_inventory_fields_edit');
    const stockInput = document.getElementById('stock_input_edit');
    const shippingInput = document.getElementById('shipping_fee_input_edit');
    const syncHardcopy = () => {
        if (!hardcopyPrice || !hardcopyFields) return;
        if (hardcopyToggle?.checked) {
            hardcopyPrice.classList.remove('hidden');
            hardcopyPrice.removeAttribute('disabled');
            hardcopyPrice.required = true;
            hardcopyFields.classList.remove('hidden');
            stockInput?.removeAttribute('disabled');
            stockInput?.setAttribute('required', 'required');
            shippingInput?.removeAttribute('disabled');
        } else {
            hardcopyPrice.classList.add('hidden');
            hardcopyPrice.setAttribute('disabled', 'disabled');
            hardcopyPrice.required = false;
            hardcopyPrice.value = '';
            hardcopyFields.classList.add('hidden');
            if (stockInput) {
                stockInput.value = '';
                stockInput.setAttribute('disabled', 'disabled');
                stockInput.removeAttribute('required');
            }
            if (shippingInput) {
                shippingInput.value = '';
                shippingInput.setAttribute('disabled', 'disabled');
            }
        }
    };
    hardcopyToggle?.addEventListener('change', syncHardcopy);
    syncHardcopy();
});
</script>
@endsection
