<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Cache;
use App\Http\Controllers\HomeController;
use App\Http\Controllers\BookController;
use App\Http\Controllers\CartController;
use App\Http\Controllers\CheckoutController;
use App\Http\Controllers\BlogController;
use App\Http\Controllers\PromotionController;
use App\Http\Controllers\NewsletterController;
use App\Http\Controllers\ContactController;
use App\Http\Controllers\WebhookController;
use App\Http\Controllers\Admin\SiteController;
use App\Http\Controllers\ProfileController;
use App\Http\Controllers\Auth\SocialController;
use App\Http\Controllers\DownloadController;

Route::get('/', function () {
    return app(HomeController::class)();
})->name('home');

// Static pages
Route::view('/about', 'about')->name('about');
Route::view('/services', 'services')->name('services');
Route::view('/merch', 'merch')->name('merch');
Route::get('/downloads/{token}', [DownloadController::class, 'show'])->name('downloads.show');

// Books
Route::get('/books', [BookController::class, 'index'])->name('books.index');
Route::get('/books/{slug}', [BookController::class, 'show'])->name('books.show');

// Cart
Route::get('/cart', [CartController::class, 'index'])->name('cart.index');
Route::post('/cart/{slug}', [CartController::class, 'add'])->name('cart.add');
Route::patch('/cart', [CartController::class, 'update'])->name('cart.update');
Route::delete('/cart', [CartController::class, 'remove'])->name('cart.remove');

// Checkout
Route::get('/checkout', [CheckoutController::class, 'show'])->name('checkout.show');
Route::post('/checkout/intent', [CheckoutController::class, 'intent'])->name('checkout.intent');
Route::post('/checkout/stripe', [CheckoutController::class, 'stripe'])->name('checkout.stripe');
Route::post('/checkout/paypal', [CheckoutController::class, 'paypal'])->name('checkout.paypal');
Route::get('/checkout/verify', [CheckoutController::class, 'verify'])->name('checkout.verify');

// Blog
Route::get('/blog', [BlogController::class, 'index'])->name('blog.index');
Route::get('/blog/{slug}', [BlogController::class, 'show'])->name('blog.show');
Route::post('/blog/{slug}/comments', [BlogController::class, 'comment'])->name('blog.comment');

// Promotions
Route::get('/promotions', [PromotionController::class, 'index'])->name('promotions.index');
Route::get('/promotions/{code}', [PromotionController::class, 'show'])->name('promotions.show');
Route::post('/promotions/{code}/entries', [PromotionController::class, 'enter'])->name('promotions.enter');

// Newsletter
Route::post('/newsletter/subscribe', [NewsletterController::class, 'subscribe'])->name('newsletter.subscribe');
Route::get('/newsletter/confirm/{token}', [NewsletterController::class, 'confirm'])->name('newsletter.confirm');
Route::get('/newsletter/unsubscribe/{token}', [NewsletterController::class, 'unsubscribe'])->name('newsletter.unsubscribe');

// Contact
Route::get('/contact', [ContactController::class, 'show'])->name('contact.show');
Route::post('/contact', [ContactController::class, 'send'])->name('contact.send');

// Webhooks
Route::post('/webhooks/stripe', [WebhookController::class, 'stripe']);
Route::post('/webhooks/paypal', [WebhookController::class, 'paypal']);

// Authenticated user profile (Breeze)
Route::get('/dashboard', function () {
    return view('dashboard');
})->middleware(['auth', 'verified'])->name('dashboard');

Route::middleware('auth')->group(function () {
    Route::get('/profile', [ProfileController::class, 'edit'])->name('profile.edit');
    Route::patch('/profile', [ProfileController::class, 'update'])->name('profile.update');
    Route::delete('/profile', [ProfileController::class, 'destroy'])->name('profile.destroy');
});

require __DIR__.'/auth.php';

// Admin area (admin and cashier access, both must be authenticated and verified)
Route::middleware(['auth', 'verified', 'role:admin,cashier'])->prefix('admin')->group(function () {
    Route::get('/', \App\Http\Controllers\Admin\DashboardController::class)->name('admin.dashboard');

    // Products (admin only enforced by policies)
    Route::resource('products', \App\Http\Controllers\Admin\ProductController::class)->except(['show'])->names('admin.products');
    // Categories (admin only enforced by policies)
    Route::resource('categories', \App\Http\Controllers\Admin\CategoryController::class)->except(['show'])->names('admin.categories');
    Route::get('categories/slug', [\App\Http\Controllers\Admin\CategoryController::class, 'slug'])->name('admin.categories.slug');

    // Orders
    Route::get('orders', [\App\Http\Controllers\Admin\OrderController::class, 'index'])->name('admin.orders.index');
    Route::get('orders/{order}', [\App\Http\Controllers\Admin\OrderController::class, 'show'])->name('admin.orders.show');
    Route::patch('orders/{order}/status', [\App\Http\Controllers\Admin\OrderController::class, 'updateStatus'])->name('admin.orders.status');

    // Posts
    Route::resource('posts', \App\Http\Controllers\Admin\PostController::class)->except(['show'])->names('admin.posts');

    // Comments moderation
    Route::get('comments', [\App\Http\Controllers\Admin\CommentController::class, 'index'])->name('admin.comments.index');
    Route::patch('comments/{comment}/status', [\App\Http\Controllers\Admin\CommentController::class, 'updateStatus'])->name('admin.comments.status');
    Route::delete('comments/{comment}', [\App\Http\Controllers\Admin\CommentController::class, 'destroy'])->name('admin.comments.destroy');

    // Admin tools (restricted to admins only)
    Route::middleware('role:admin')->group(function () {
        Route::get('settings', [\App\Http\Controllers\Admin\SettingsController::class, 'edit'])->name('admin.settings.edit');
        Route::get('settings/site', [\App\Http\Controllers\Admin\SiteController::class, 'edit'])->name('admin.site.edit');
        Route::post('settings/site', [\App\Http\Controllers\Admin\SiteController::class, 'update'])->name('admin.site.update');
        Route::post('settings', [\App\Http\Controllers\Admin\SettingsController::class, 'update'])->name('admin.settings.update');
        Route::post('settings/test/{provider}', [\App\Http\Controllers\Admin\SettingsController::class, 'test'])->name('admin.settings.test');
        Route::resource('newsletters', \App\Http\Controllers\Admin\NewsletterCampaignController::class)->except(['show', 'edit'])->names('admin.newsletters');
    });
});









// OAuth: Social providers (Google, Facebook, GitHub, Twitter, LinkedIn)
Route::get('/auth/{provider}/redirect', [SocialController::class, 'redirect'])
    ->whereIn('provider', ['google','facebook','github','twitter','linkedin'])
    ->name('oauth.redirect');
Route::get('/auth/{provider}/callback', [SocialController::class, 'callback'])
    ->whereIn('provider', ['google','facebook','github','twitter','linkedin'])
    ->name('oauth.callback');

Route::get('/sitemap.xml', function () {
    $urls = Cache::remember('sitemap.xml', 3600, function () {
        $entries = [];

        $staticRoutes = [
            ['route' => 'home', 'changefreq' => 'daily', 'priority' => '1.0'],
            ['route' => 'about', 'changefreq' => 'monthly'],
            ['route' => 'books.index', 'changefreq' => 'daily'],
            ['route' => 'promotions.index', 'changefreq' => 'weekly'],
            ['route' => 'services', 'changefreq' => 'monthly'],
            ['route' => 'blog.index', 'changefreq' => 'weekly'],
            ['route' => 'merch', 'changefreq' => 'monthly'],
            ['route' => 'contact.show', 'changefreq' => 'monthly'],
        ];

        foreach ($staticRoutes as $routeConfig) {
            if (!Route::has($routeConfig['route'])) {
                continue;
            }
            $entries[] = [
                'loc' => route($routeConfig['route']),
                'lastmod' => now(),
                'changefreq' => $routeConfig['changefreq'] ?? 'weekly',
                'priority' => $routeConfig['priority'] ?? '0.7',
            ];
        }

        $products = \App\Models\Product::query()
            ->whereNotNull('published_at')
            ->orderByDesc('updated_at')
            ->limit(500)
            ->get(['slug', 'updated_at', 'published_at']);

        foreach ($products as $product) {
            $entries[] = [
                'loc' => route('books.show', $product->slug),
                'lastmod' => $product->updated_at ?? $product->published_at,
                'changefreq' => 'weekly',
                'priority' => '0.6',
            ];
        }

        $posts = \App\Models\Post::query()
            ->where('is_published', true)
            ->orderByDesc('updated_at')
            ->limit(500)
            ->get(['slug', 'updated_at', 'published_at']);

        foreach ($posts as $post) {
            $entries[] = [
                'loc' => route('blog.show', $post->slug),
                'lastmod' => $post->updated_at ?? $post->published_at,
                'changefreq' => 'weekly',
                'priority' => '0.6',
            ];
        }

        $promotions = \App\Models\Promotion::query()
            ->where('is_active', true)
            ->orderByDesc('updated_at')
            ->limit(200)
            ->get(['code', 'updated_at', 'starts_at', 'ends_at']);

        foreach ($promotions as $promotion) {
            $entries[] = [
                'loc' => route('promotions.show', $promotion->code),
                'lastmod' => $promotion->updated_at ?? $promotion->ends_at ?? $promotion->starts_at,
                'changefreq' => 'weekly',
                'priority' => '0.5',
            ];
        }

        return $entries;
    });

    return response()->view('sitemap', ['urls' => $urls])->header('Content-Type', 'application/xml');
});



