<?php

namespace Tests\Feature;

use App\Models\Order;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Http;
use Tests\TestCase;

class WebhookTest extends TestCase
{
    use RefreshDatabase;

    public function test_stripe_webhook_valid_signature_marks_order_paid()
    {
        $order = Order::factory()->create([
            'status' => 'pending',
            'amount_kobo' => 1234,
            'reference' => 'ord_'.Str::uuid(),
        ]);

        $payload = json_encode([
            'id' => 'evt_123',
            'type' => 'checkout.session.completed',
            'data' => [
                'object' => [
                    'amount_total' => 1234,
                    'currency' => 'usd',
                    'metadata' => [
                        'order_reference' => $order->reference,
                    ],
                ],
            ],
        ]);

        config(['services.stripe.webhook_secret' => 'whsec_test']);
        $timestamp = time();
        $sig = hash_hmac('sha256', $timestamp.'.'.$payload, 'whsec_test');

        $res = $this->withHeaders([
            'Stripe-Signature' => 't='.$timestamp.',v1='.$sig,
            'Content-Type' => 'application/json',
        ])->postJson('/webhooks/stripe', json_decode($payload, true));

        $res->assertOk();
        $this->assertEquals('paid', $order->fresh()->status);
    }

    public function test_stripe_webhook_invalid_signature_rejected()
    {
        $payload = json_encode(['id' => 'evt_bad', 'type' => 'checkout.session.completed', 'data' => ['object' => []]]);
        config(['services.stripe.webhook_secret' => 'whsec_test']);
        $res = $this->withHeaders([
            'Stripe-Signature' => 't=1,v1=bad',
            'Content-Type' => 'application/json',
        ])->postJson('/webhooks/stripe', json_decode($payload, true));
        $res->assertStatus(400);
    }

    public function test_paypal_webhook_verification_marks_order_paid()
    {
        Http::fake([
            '*/v1/oauth2/token' => Http::response(['access_token' => 'tok'], 200),
            '*/v1/notifications/verify-webhook-signature' => Http::response(['verification_status' => 'SUCCESS'], 200),
        ]);

        config(['services.paypal.webhook_id' => 'wh-123']);

        $order = Order::factory()->create([
            'status' => 'pending',
            'amount_kobo' => 5000,
            'reference' => 'ord_'.Str::uuid(),
        ]);

        $event = [
            'id' => 'WH-evt-1',
            'event_type' => 'PAYMENT.CAPTURE.COMPLETED',
            'resource' => [
                'amount' => [
                    'value' => '50.00',
                    'currency_code' => 'USD',
                ],
                'supplementary_data' => [
                    'related_ids' => [
                        'order_id' => $order->reference,
                    ],
                ],
            ],
        ];

        $res = $this->withHeaders([
            'paypal-auth-algo' => 'fake',
            'paypal-transmission-id' => 'tid',
            'paypal-cert-url' => 'https://example.com/cert',
            'paypal-transmission-sig' => 'sig',
            'paypal-transmission-time' => now()->toRfc3339String(),
            'Content-Type' => 'application/json',
        ])->postJson('/webhooks/paypal', $event);

        $res->assertOk();
        $this->assertEquals('paid', $order->fresh()->status);
    }

    public function test_paypal_webhook_failed_verification_rejected()
    {
        Http::fake([
            '*/v1/oauth2/token' => Http::response(['access_token' => 'tok'], 200),
            '*/v1/notifications/verify-webhook-signature' => Http::response(['verification_status' => 'FAILURE'], 200),
        ]);
        config(['services.paypal.webhook_id' => 'wh-123']);
        $res = $this->withHeaders([
            'paypal-auth-algo' => 'fake',
            'paypal-transmission-id' => 'tid',
            'paypal-cert-url' => 'https://example.com/cert',
            'paypal-transmission-sig' => 'sig',
            'paypal-transmission-time' => now()->toRfc3339String(),
            'Content-Type' => 'application/json',
        ])->postJson('/webhooks/paypal', ['id' => 'x', 'event_type' => 'PAYMENT.CAPTURE.COMPLETED']);
        $res->assertStatus(400);
    }
}