<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CategoryController extends Controller
{
    public function index()
    {
        $categories = Category::query()->withCount('products')->orderBy('name')->paginate(20);
        return view('admin.categories.index', compact('categories'));
    }

    public function create()
    {
        $this->authorize('create', Category::class);
        return view('admin.categories.create');
    }

    public function store(Request $request)
    {
        $this->authorize('create', Category::class);
        $data = $request->validate([
            'name' => ['required','string','max:120'],
            'slug' => ['nullable','string','max:150'],
        ]);
        $category = Category::create($data);
        \App\Services\AuditService::log('category.created', $category, $request->user());
        return redirect()->route('admin.categories.index')->with('status','Category created');
    }

    public function edit(Category $category)
    {
        $this->authorize('update', $category);
        return view('admin.categories.edit', compact('category'));
    }

    public function update(Request $request, Category $category)
    {
        $this->authorize('update', $category);
        $data = $request->validate([
            'name' => ['required','string','max:120'],
            'slug' => ['nullable','string','max:150'],
        ]);
        $category->update($data);
        \App\Services\AuditService::log('category.updated', $category, $request->user());
        return redirect()->route('admin.categories.index')->with('status','Category updated');
    }

    public function destroy(Category $category, Request $request)
    {
        $this->authorize('delete', $category);
        if ($category->products()->exists()) {
            return back()->with('error', 'Cannot delete a category that has products');
        }
        $category->delete();
        \App\Services\AuditService::log('category.deleted', $category, $request->user());
        return back()->with('status','Category deleted');
    }

    // AJAX: generate unique slug suggestion from name
    public function slug(Request $request)
    {
        $name = (string) $request->get('name', '');
        $base = Str::slug($name);
        $slug = $base;
        $i = 1;
        while ($slug && Category::where('slug', $slug)->exists()) {
            $slug = $base.'-'.$i++;
        }
        return response()->json(['slug' => $slug, 'unique' => $slug !== '' && !Category::where('slug',$slug)->exists()]);
    }
}
