<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ProductController extends Controller
{
    public function index(\Illuminate\Http\Request $request)
    {
        // Authorization handled by admin route middleware (role: admin,cashier)
        $filter = $request->string('filter'); // latest|discounted|featured
        $categoryId = $request->integer('category_id');
        $q = $request->string('q');
        $low = $request->boolean('low');

        $products = Product::query()
            ->when($q, fn($x) => $x->where('title','like','%'.$q.'%'))
            ->when($categoryId, fn($x) => $x->where('category_id', $categoryId))
            ->when($filter === 'discounted', fn($x) => $x->discounted())
            ->when($filter === 'featured', fn($x) => $x->where('is_featured', true))
            ->when($filter === 'latest' || !$filter, fn($x) => $x->latest('published_at'))
            ->when($low, fn($x) => $x->where('stock','<=',5))
            ->with('category')
            ->paginate(15)
            ->withQueryString();
        $products->withPath(route('admin.products.index'));

        $categories = \App\Models\Category::orderBy('name')->get(['id','name']);
        return view('admin.products.index', compact('products','categories','filter','categoryId','q','low'));
    }

    public function create()
    {
        $this->authorize('create', Product::class);
        return view('admin.products.create');
    }

    public function store(Request $request)
    {
        $this->authorize('create', Product::class);
        $data = $request->validate([
            'title' => 'required|string|max:255',
            'type' => 'required|string|max:50',
            'excerpt' => 'nullable|string',
            'description' => 'nullable|string',
            'price_amount' => 'required|numeric|min:0',
            'sale_price_amount' => 'nullable|numeric|min:0|lt:price_amount',
            'stock' => 'nullable|integer|min:0',
            'has_hardcopy' => 'nullable|boolean',
            'hardcopy_price_amount' => 'nullable|numeric|min:0',
            'shipping_fee_amount' => 'nullable|numeric|min:0',
            'cover_image' => 'nullable|image|max:2048',
            'download_file' => 'nullable|file|max:51200',
            'is_featured' => 'boolean',
            'published_at' => 'nullable|date',
            'category_id' => 'nullable|integer|exists:categories,id',
            'category_name' => 'nullable|string|max:120',
        ]);
        $hasHardcopy = $request->boolean('has_hardcopy');
        if ($hasHardcopy && empty($data['hardcopy_price_amount'])) {
            return back()->withInput()->withErrors(['hardcopy_price_amount' => 'Provide a price for the hardcopy edition.']);
        }
        // Normalize optional fields to satisfy DB constraints
        $data['stock'] = $hasHardcopy ? ($data['stock'] ?? 0) : 0;
        $data['is_featured'] = $request->boolean('is_featured');
        $data['has_hardcopy'] = $hasHardcopy;
        // Convert dollars to minor units (cents) for storage
        $data['price_kobo'] = (int) round(((float) $data['price_amount']) * 100);
        if (!empty($data['sale_price_amount'])) {
            $data['sale_price_kobo'] = (int) round(((float) $data['sale_price_amount']) * 100);
        }
        $shippingAmount = $request->input('shipping_fee_amount');
        $data['shipping_fee_kobo'] = $hasHardcopy && $shippingAmount !== null ? (int) round(((float) $shippingAmount) * 100) : 0;
        $hardcopyPrice = $request->input('hardcopy_price_amount');
        $data['hardcopy_price_kobo'] = $hasHardcopy ? (int) round(((float) $hardcopyPrice) * 100) : null;
        unset($data['price_amount'], $data['sale_price_amount'], $data['shipping_fee_amount'], $data['hardcopy_price_amount']);
        if ($request->hasFile('cover_image')) {
            // Store on the public disk so it is served via /storage
            $path = $request->file('cover_image')->store('covers', 'public');
            $data['cover_path'] = 'storage/'.$path;
        }
        if ($request->hasFile('download_file')) {
            $path = $request->file('download_file')->store('products/downloads', 'local');
            $data['download_path'] = $path;
            $data['download_disk'] = 'local';
        }
        if (empty($data['category_id']) && !empty($data['category_name'])) {
            $cat = \App\Models\Category::firstOrCreate(['name' => $data['category_name']]);
            $data['category_id'] = $cat->id;
        }
        unset($data['category_name']);
        $product = Product::create($data);
        \App\Services\AuditService::log('product.created', $product, $request->user());
        return redirect()->route('admin.products.index')->with('status','Product created');
    }

    public function edit(Product $product)
    {
        $this->authorize('update', $product);
        return view('admin.products.edit', compact('product'));
    }

    public function update(Request $request, Product $product)
    {
        $this->authorize('update', $product);
        $data = $request->validate([
            'title' => 'required|string|max:255',
            'type' => 'required|string|max:50',
            'excerpt' => 'nullable|string',
            'description' => 'nullable|string',
            'price_amount' => 'required|numeric|min:0',
            'sale_price_amount' => 'nullable|numeric|min:0|lt:price_amount',
            'stock' => 'nullable|integer|min:0',
            'has_hardcopy' => 'nullable|boolean',
            'hardcopy_price_amount' => 'nullable|numeric|min:0',
            'shipping_fee_amount' => 'nullable|numeric|min:0',
            'cover_image' => 'nullable|image|max:2048',
            'download_file' => 'nullable|file|max:51200',
            'remove_download' => 'nullable|boolean',
            'is_featured' => 'boolean',
            'published_at' => 'nullable|date',
            'category_id' => 'nullable|integer|exists:categories,id',
            'category_name' => 'nullable|string|max:120',
        ]);
        $hasHardcopy = $request->boolean('has_hardcopy');
        if ($hasHardcopy && empty($data['hardcopy_price_amount'])) {
            return back()->withInput()->withErrors(['hardcopy_price_amount' => 'Provide a price for the hardcopy edition.']);
        }
        $data['price_kobo'] = (int) round(((float) $data['price_amount']) * 100);
        if (!empty($data['sale_price_amount'])) {
            $data['sale_price_kobo'] = (int) round(((float) $data['sale_price_amount']) * 100);
        } else {
            $data['sale_price_kobo'] = null;
        }
        $data['has_hardcopy'] = $hasHardcopy;
        $data['stock'] = $hasHardcopy ? ($data['stock'] ?? 0) : 0;
        $hardcopyPrice = $request->input('hardcopy_price_amount');
        $data['hardcopy_price_kobo'] = $hasHardcopy ? (int) round(((float) $hardcopyPrice) * 100) : null;
        unset($data['price_amount'], $data['sale_price_amount']);
        $shippingAmount = $request->input('shipping_fee_amount');
        $data['shipping_fee_kobo'] = $hasHardcopy && $shippingAmount !== null ? (int) round(((float) $shippingAmount) * 100) : 0;
        unset($data['shipping_fee_amount'], $data['hardcopy_price_amount']);
        if ($request->hasFile('cover_image')) {
            $path = $request->file('cover_image')->store('covers', 'public');
            $data['cover_path'] = 'storage/'.$path;
        }
        $removeDownload = $request->boolean('remove_download');
        if ($request->hasFile('download_file')) {
            if ($product->download_path) {
                $disk = $product->download_disk ?: 'local';
                Storage::disk($disk)->delete($product->download_path);
            }
            $path = $request->file('download_file')->store('products/downloads', 'local');
            $data['download_path'] = $path;
            $data['download_disk'] = 'local';
        } elseif ($removeDownload && $product->download_path) {
            $disk = $product->download_disk ?: 'local';
            Storage::disk($disk)->delete($product->download_path);
            $data['download_path'] = null;
            $data['download_disk'] = null;
        }
        if (empty($data['category_id']) && !empty($data['category_name'])) {
            $cat = \App\Models\Category::firstOrCreate(['name' => $data['category_name']]);
            $data['category_id'] = $cat->id;
        }
        unset($data['category_name']);
        $product->update($data);
        \App\Services\AuditService::log('product.updated', $product, $request->user());
        return redirect()->route('admin.products.index')->with('status','Product updated');
    }

    public function destroy(Product $product, Request $request)
    {
        $this->authorize('delete', $product);
        $product->delete();
        \App\Services\AuditService::log('product.deleted', $product, $request->user());
        return back()->with('status','Product deleted');
    }
}
