<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use App\Services\Payments\PaypalGateway;
use App\Services\Payments\StripeGateway;
use Illuminate\Http\Request;

class SettingsController extends Controller
{
    public function edit()
    {
        $values = [
            'stripe_secret' => Setting::get('stripe.secret'),
            'stripe_public' => Setting::get('stripe.public'),
            'stripe_base_url' => Setting::get('stripe.base_url') ?? 'https://api.stripe.com',
            'paypal_client_id' => Setting::get('paypal.client_id'),
            'paypal_secret' => Setting::get('paypal.secret'),
            'paypal_base_url' => Setting::get('paypal.base_url') ?? 'https://api-m.paypal.com',
            'create_test_order' => (bool) Setting::get('payments.create_test_order', '1'),
        ];
        return view('admin.settings.edit', compact('values'));
    }

    public function update(Request $request)
    {
        $data = $request->validate([
            'stripe_secret' => 'nullable|string',
            'stripe_public' => 'nullable|string',
            'stripe_base_url' => 'nullable|string',
            'paypal_client_id' => 'nullable|string',
            'paypal_secret' => 'nullable|string',
            'paypal_base_url' => 'nullable|string',
            'create_test_order' => 'nullable|boolean',
        ]);

        Setting::set('stripe.secret', $data['stripe_secret'] ?? null);
        Setting::set('stripe.public', $data['stripe_public'] ?? null);
        Setting::set('stripe.base_url', $data['stripe_base_url'] ?? null);
        Setting::set('paypal.client_id', $data['paypal_client_id'] ?? null);
        Setting::set('paypal.secret', $data['paypal_secret'] ?? null);
        Setting::set('paypal.base_url', $data['paypal_base_url'] ?? null);
        Setting::set('payments.create_test_order', ($data['create_test_order'] ?? false) ? '1' : '0');
        \App\Models\Setting::flushAllCache();

        \App\Services\AuditService::log('settings.updated', 'payments', $request->user());
        return back()->with('status', 'Settings saved');
    }

    public function test(Request $request, string $provider)
    {
        $user = $request->user();
        $email = $user->email;
        $amount = 1000; // $10.00 test charge in cents

        $reference = 'test_'.\Illuminate\Support\Str::uuid();
        $orderMeta = [
            'source' => 'admin_test_payment',
            'is_test' => true,
            'initiated_by' => $user->id,
            'provider' => $provider,
            'reference' => $reference,
        ];

        if ((bool) Setting::get('payments.create_test_order', '1')) {
            \App\Models\Order::create([
                'user_id' => $user->id,
                'email' => $email,
                'name' => $user->name,
                'status' => 'pending',
                'amount_kobo' => $amount,
                'currency' => config('payment.currency', 'USD'),
                'address_json' => null,
                'reference' => $reference,
                'metadata' => $orderMeta,
            ]);
        }

        if ($provider === 'stripe') {
            $init = app(StripeGateway::class)->init($amount, $email, $orderMeta + [
                'reference' => $reference,
                'success_url' => route('checkout.verify').'?provider=stripe&reference='.$reference.'&session_id={CHECKOUT_SESSION_ID}',
                'cancel_url' => route('admin.settings.edit'),
            ]);
            return redirect()->away($init['checkout_url']);
        }

        if ($provider === 'paypal') {
            $init = app(PaypalGateway::class)->init($amount, $email, $orderMeta + [
                'reference' => $reference,
                'return_url' => route('checkout.verify').'?provider=paypal&reference='.$reference,
                'cancel_url' => route('admin.settings.edit'),
            ]);
            return redirect()->away($init['approve_url']);
        }

        return back()->with('error', 'Unknown provider');
    }
}
