<?php

namespace App\Http\Controllers;

use App\Http\Requests\AddToCartRequest;
use App\Models\Product;
use App\Services\CartService;
use Illuminate\Http\Request;

class CartController extends Controller
{
    public function __construct(private CartService $cart) {}

    public function index()
    {
        $items = $this->cart->items();
        $subTotal = array_sum(array_map(function ($item) {
            return (int) ($item['price_kobo'] ?? 0) * (int) ($item['qty'] ?? 0);
        }, $items));

        return view('cart', [
            'items' => $items,
            'subtotalKobo' => $subTotal,
            'shippingKobo' => $this->cart->shippingKobo(),
            'totalKobo' => $this->cart->totalKobo(),
        ]);
    }

    public function add(AddToCartRequest $request, string $slug)
    {
        $product = Product::where('slug', $slug)->firstOrFail();
        $variant = $request->validated('variant') ?? 'digital';
        $quantity = $variant === 'hardcopy' ? (int) $request->validated('qty') : 1;
        $this->cart->add($product, $variant, $quantity);
        return redirect()->route('cart.index')->with('status', 'Added to cart');
    }

    public function update(Request $request)
    {
        $data = $request->validate([
            'product_id' => ['required','integer'],
            'variant' => ['required','string','in:digital,hardcopy'],
            'qty' => ['required_if:variant,hardcopy','integer','min:0','max:99'],
        ]);
        $qty = ($data['variant'] ?? '') === 'hardcopy' ? (int) $data['qty'] : 1;
        $this->cart->update((int) $data['product_id'], $data['variant'], $qty);
        return back();
    }

    public function remove(Request $request)
    {
        $data = $request->validate([
            'product_id' => ['required','integer'],
            'variant' => ['required','string','in:digital,hardcopy'],
        ]);
        $this->cart->remove((int) $data['product_id'], $data['variant']);
        return back();
    }
}
