<?php

namespace App\Http\Controllers;

use App\Http\Requests\CheckoutIntentRequest;
use App\Models\Order;
use App\Models\OrderItem;
use App\Services\CartService;
use App\Services\OrderFulfillmentService;
use App\Services\Payments\PaypalGateway;
use App\Services\Payments\StripeGateway;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class CheckoutController extends Controller
{
    public function __construct(private CartService $cart)
    {
    }

    public function show()
    {
        $items = $this->cart->items();
        $subTotal = array_sum(array_map(function ($item) {
            return (int) ($item['price_kobo'] ?? 0) * (int) ($item['qty'] ?? 0);
        }, $items));

        return view('checkout', [
            'cart' => [
                'items' => $items,
                'subtotalKobo' => $subTotal,
                'shippingKobo' => $this->cart->shippingKobo(),
                'totalKobo' => $this->cart->totalKobo(),
            ],
            'currency' => config('payment.currency', 'USD'),
            'currencySymbol' => config('payment.currency_symbol', '$'),
        ]);
    }

    public function intent(CheckoutIntentRequest $request)
    {
        $data = $request->validated();
        $payload = $this->cart->toOrderPayload();

        if ($payload['amount_kobo'] <= 0) {
            return response()->json(['message' => 'Cart is empty'], 422);
        }

        $order = DB::transaction(function () use ($data, $payload) {
            $order = Order::create([
                'user_id' => auth()->id(),
                'email' => $data['email'],
                'name' => $data['name'],
                'status' => 'pending',
                'amount_kobo' => $payload['amount_kobo'],
                'shipping_kobo' => $payload['shipping_kobo'],
                'currency' => config('payment.currency', 'USD'),
                'address_json' => $data['address'],
                'reference' => 'ord_' . Str::uuid(),
                'metadata' => ['source' => 'checkout_intent'],
            ]);

            foreach ($payload['items'] as $item) {
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $item['product_id'] ?? null,
                    'product_type' => $item['product_type'] ?? null,
                    'is_digital' => (bool) ($item['is_digital'] ?? false),
                    'title' => $item['title'],
                    'unit_price_kobo' => $item['price_kobo'],
                    'shipping_fee_kobo' => $item['shipping_fee_kobo'] ?? 0,
                    'qty' => $item['qty'],
                    'total_kobo' => $item['price_kobo'] * $item['qty'],
                ]);
            }

            return $order;
        });

        return response()->json([
            'order_id' => $order->id,
            'reference' => $order->reference,
        ]);
    }

    public function stripe(Request $request, StripeGateway $stripe)
    {
        $data = $request->validate([
            'order_id' => ['required', 'integer', 'exists:orders,id'],
        ]);

        $order = Order::findOrFail($data['order_id']);

        $init = $stripe->init($order->amount_kobo, $order->email, [
            'order_id' => $order->id,
            'reference' => $order->reference,
            'description' => 'Order ' . $order->reference,
            'success_url' => route('checkout.verify') . '?provider=stripe&reference=' . urlencode($order->reference) . '&session_id={CHECKOUT_SESSION_ID}',
            'cancel_url' => route('cart.index'),
        ]);

        return response()->json($init);
    }

    public function paypal(Request $request, PaypalGateway $paypal)
    {
        $data = $request->validate([
            'order_id' => ['required', 'integer', 'exists:orders,id'],
        ]);

        $order = Order::findOrFail($data['order_id']);

        $init = $paypal->init($order->amount_kobo, $order->email, [
            'order_id' => $order->id,
            'reference' => $order->reference,
            'return_url' => route('checkout.verify') . '?provider=paypal&reference=' . urlencode($order->reference),
            'cancel_url' => route('cart.index'),
        ]);

        return response()->json($init);
    }

    public function verify(Request $request)
    {
        $provider = (string) $request->string('provider');
        $reference = (string) $request->string('reference');

        $order = $reference !== ''
            ? Order::where('reference', $reference)->first()
            : null;

        $verifiedReference = null;
        $amountMinor = 0;
        $ok = false;

        if ($provider === 'stripe') {
            $sessionId = (string) $request->string('session_id');
            $res = app(StripeGateway::class)->verify($sessionId);
            $ok = ($res['status'] ?? '') === 'success';
            $amountMinor = (int) ($res['amount_cents'] ?? 0);
            $verifiedReference = $res['reference'] ?? null;
        } elseif ($provider === 'paypal') {
            $orderId = (string) $request->string('token');
            $res = app(PaypalGateway::class)->verify($orderId);
            $ok = ($res['status'] ?? '') === 'success';
            $amountMinor = (int) ($res['amount_cents'] ?? 0);
            $verifiedReference = $res['reference'] ?? null;
        } else {
            return redirect()->route('cart.index')->with('error', 'Unknown payment provider');
        }

        if (!$order && $verifiedReference) {
            $order = Order::where('reference', $verifiedReference)->first();
        }

        if (!$order) {
            return redirect()->route('cart.index')->with('error', 'Order not found');
        }

        if ($ok && $order->status !== 'paid') {
            if ($amountMinor > 0 && (int) $order->amount_kobo !== $amountMinor) {
                // Amount mismatch: keep order pending for manual review.
            } else {
                $order->update([
                    'status' => 'paid',
                    'placed_at' => now(),
                    'provider' => $provider,
                ]);
                $order->refresh();
            }
        }

        if ($order->status === 'paid') {
            app(OrderFulfillmentService::class)->handlePaidOrder($order);
        }

        $this->cart->clear();

        return view('checkout-success', compact('order'));
    }
}
