<?php

namespace App\Http\Controllers;

use App\Mail\NewsletterConfirmationMail;
use App\Models\Subscriber;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\View\View;

class NewsletterController extends Controller
{
    public function subscribe(Request $request): RedirectResponse
    {
        $data = $request->validate([
            'email' => ['required', 'email'],
        ]);

        $email = strtolower($data['email']);
        $source = $request->input('source', 'site');

        $subscriber = Subscriber::firstOrNew(['email' => $email]);

        if ($subscriber->exists && $subscriber->confirmed_at && is_null($subscriber->unsubscribed_at)) {
            return back()->with('status', 'You are already subscribed to the newsletter.');
        }

        $subscriber->refreshToken();
        $subscriber->source = $source;
        $subscriber->confirmed_at = null;
        $subscriber->unsubscribed_at = null;
        $subscriber->save();

        Mail::to($subscriber->email)->send(new NewsletterConfirmationMail($subscriber));

        return back()->with('status', 'Thanks! Please check your email to confirm your subscription.');
    }

    public function confirm(string $token): View|RedirectResponse
    {
        $subscriber = Subscriber::where('token', $token)->first();

        if (! $subscriber) {
            return redirect()->route('home')->with('toast', 'Confirmation link is invalid or has expired.');
        }

        $subscriber->markConfirmed();
        $subscriber->refreshToken();
        $subscriber->save();

        return view('newsletter.confirmed', ['subscriber' => $subscriber]);
    }

    public function unsubscribe(string $token): View|RedirectResponse
    {
        $subscriber = Subscriber::where('token', $token)->first();

        if (! $subscriber) {
            return redirect()->route('home')->with('toast', 'We could not process your unsubscribe request.');
        }

        $subscriber->markUnsubscribed();
        $subscriber->save();

        return view('newsletter.unsubscribed', ['subscriber' => $subscriber]);
    }
}

