<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use Illuminate\Support\Carbon;

class Subscriber extends Model
{
    /**
     * Allow mass assignment.
     */
    protected $guarded = [];

    /**
     * Cast date columns to Carbon instances.
     */
    protected $casts = [
        'confirmed_at' => 'datetime',
        'unsubscribed_at' => 'datetime',
    ];

    /**
     * Scope confirmed subscribers.
     */
    public function scopeConfirmed(Builder $query): Builder
    {
        return $query->whereNotNull('confirmed_at');
    }

    /**
     * Scope active (not unsubscribed) subscribers.
     */
    public function scopeActive(Builder $query): Builder
    {
        return $query->whereNull('unsubscribed_at');
    }

    /**
     * Generate or refresh the unique token used for confirmation/unsubscribe.
     */
    public function refreshToken(): void
    {
        $this->token = static::generateUniqueToken();
    }

    /**
     * Create a unique random token.
     */
    public static function generateUniqueToken(): string
    {
        do {
            $token = Str::random(40);
        } while (static::where('token', $token)->exists());

        return $token;
    }

    /**
     * Mark the subscriber as confirmed.
     */
    public function markConfirmed(): void
    {
        $this->confirmed_at = Carbon::now();
        $this->unsubscribed_at = null;
    }

    /**
     * Soft unsubscribe the subscriber.
     */
    public function markUnsubscribed(): void
    {
        $this->unsubscribed_at = Carbon::now();
    }
}
