<?php

namespace App\Services;

use App\Mail\OrderDigitalFulfillmentMail;
use App\Mail\OrderShippingNotificationMail;
use App\Models\Download;
use App\Models\Order;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;

class OrderFulfillmentService
{
    public function handlePaidOrder(Order $order): void
    {
        $order->loadMissing('items.product');

        $metadata = $order->metadata ?? [];
        $digitalLinks = $this->prepareDigitalDownloads($order);
        $physicalItems = $order->items->filter(fn ($item) => !($item->is_digital ?? false));

        if (!empty($digitalLinks) && empty($metadata['digital_fulfilled_at'])) {
            Mail::to($order->email)->send(new OrderDigitalFulfillmentMail($order, $digitalLinks));
            $metadata['digital_fulfilled_at'] = now()->toIso8601String();
        }

        if ($physicalItems->isNotEmpty() && empty($metadata['shipping_notified_at'])) {
            Mail::to($order->email)->send(new OrderShippingNotificationMail($order, $physicalItems));
            $metadata['shipping_notified_at'] = now()->toIso8601String();
        }

        if ($metadata !== ($order->metadata ?? [])) {
            $order->forceFill(['metadata' => $metadata])->save();
        }
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    protected function prepareDigitalDownloads(Order $order): array
    {
        return $order->items
            ->filter(fn ($item) => ($item->is_digital ?? false) && $item->product && $item->product->download_path)
            ->map(function ($item) use ($order) {
                $product = $item->product;
                $download = Download::firstOrNew([
                    'order_id' => $order->id,
                    'product_id' => $product->id,
                ]);

                if (!$download->exists) {
                    $download->token = (string) Str::uuid();
                }

                $download->path = $product->download_path;
                $download->disk = $product->download_disk ?: 'local';
                $download->expires_at = now()->addDays(14);
                $download->save();

                return [
                    'title' => $item->title,
                    'url' => route('downloads.show', $download->token),
                    'expires_at' => $download->expires_at,
                ];
            })
            ->values()
            ->all();
    }
}
