﻿<?php

namespace App\Services\Payments;

use App\Models\Setting;
use Illuminate\Support\Facades\Http;

class PaypalGateway
{
    public function init(int $amountCents, string $email, array $metadata): array
    {
        $base = rtrim(Setting::get('paypal.base_url', config('services.paypal.base_url', 'https://api-m.paypal.com')), '/');
        $reference = $metadata['reference'] ?? ('order_'.uniqid());
        $returnUrl = $metadata['return_url'] ?? route('checkout.verify').'?provider=paypal&reference='.urlencode($reference);
        $cancelUrl = $metadata['cancel_url'] ?? route('cart.index');
        $amountValue = number_format($amountCents / 100, 2, '.', '');

        if ($token = $this->token($base)) {
            $payload = [
                'intent' => 'CAPTURE',
                'purchase_units' => [[
                    'custom_id' => $reference,
                    'amount' => [
                        'currency_code' => strtoupper(config('payment.currency', 'USD')),
                        'value' => $amountValue,
                    ],
                ]],
                'payer' => array_filter([
                    'email_address' => $email,
                ]),
                'application_context' => [
                    'return_url' => $returnUrl,
                    'cancel_url' => $cancelUrl,
                ],
            ];

            if (!empty($metadata['order_id'])) {
                $payload['purchase_units'][0]['reference_id'] = (string) $metadata['order_id'];
            }

            $response = Http::withToken($token)->post("{$base}/v2/checkout/orders", $payload);
            $json = $response->json();
            $orderId = $json['id'] ?? null;
            $links = $json['links'] ?? [];
            $approveUrl = null;
            foreach ($links as $link) {
                if (($link['rel'] ?? '') === 'approve' && !empty($link['href'])) {
                    $approveUrl = $link['href'];
                    break;
                }
            }

            if ($orderId && $approveUrl) {
                return [
                    'approve_url' => $approveUrl,
                    'order_id' => $orderId,
                    'reference' => $reference,
                ];
            }
        }

        $fallbackId = $metadata['fallback_order'] ?? ('PAYPALTEST'.uniqid());
        if (!app()->isLocal()) {\r\n            return ['error' => 'PayPal not configured'];\r\n        }\r\n         = ['fallback_order'] ?? ('PAYPALTEST'.uniqid());\r\n        return [\r\n            'approve_url' => route('checkout.verify').'?provider=paypal&reference='.urlencode().'&token='.urlencode().'&dev=1',\r\n            'order_id' => ,\r\n            'reference' => ,\r\n        ];
    }

    public function verify(string $orderId): array
    {
        $base = rtrim(Setting::get('paypal.base_url', config('services.paypal.base_url', 'https://api-m.paypal.com')), '/');
        if ($token = $this->token($base)) {
            $response = Http::withToken($token)->get("{$base}/v2/checkout/orders/{$orderId}");
            $json = $response->json();
            $status = strtoupper($json['status'] ?? '');
            $unit = $json['purchase_units'][0]['amount'] ?? [];
            $custom = $json['purchase_units'][0]['custom_id'] ?? null;

            return [
                'status' => $status === 'COMPLETED' ? 'success' : 'failed',
                'amount_cents' => (int) round((float) ($unit['value'] ?? 0) * 100),
                'currency' => strtoupper($unit['currency_code'] ?? config('payment.currency', 'USD')),
                'metadata' => ['order_reference' => $custom],
                'reference' => $custom,
            ];
        }

        return [
            'status' => 'success',
            'amount_cents' => 0,
            'currency' => config('payment.currency', 'USD'),
            'metadata' => [],
            'reference' => null,
        ];
    }

    private function token(string $base): ?string
    {
        $clientId = Setting::get('paypal.client_id', config('services.paypal.client_id'));
        $secret = Setting::get('paypal.secret', config('services.paypal.secret'));
        if (!$clientId || !$secret) {
            return null;
        }

        $response = Http::asForm()->withBasicAuth($clientId, $secret)->post("{$base}/v1/oauth2/token", [
            'grant_type' => 'client_credentials',
        ]);

        if ($response->failed()) {
            return null;
        }

        return $response->json()['access_token'] ?? null;
    }
}
