<?php

namespace App\Support;

use App\Models\Setting;
use Illuminate\Support\Arr;

class SiteSettings
{
    protected array $defaults;
    protected array $overrides = [];
    protected bool $loaded = false;

    public function __construct()
    {
        $this->defaults = config('site', []);
    }

    public function get(string $key, $fallback = null)
    {
        $default = Arr::get($this->defaults, $key, $fallback);
        $value = Arr::get($this->overrides(), $key);

        if ($value === null || (is_string($value) && trim($value) === '')) {
            return $default;
        }

        return $this->normalize($key, $value, $default);
    }

    public function appearance(): array
    {
        return $this->mergeGroup('appearance');
    }

    public function typography(): array
    {
        return $this->mergeGroup('typography');
    }
    public function brand(): array
    {
        return $this->mergeGroup('brand');
    }


    public function content(string $path = null, $fallback = null)
    {
        if ($path === null) {
            return $this->mergeGroup('content');
        }

        return $this->get('content.' . $path, $fallback);
    }

    protected function mergeGroup(string $group): array
    {
        $defaults = Arr::get($this->defaults, $group, []);
        $overrides = Arr::get($this->overrides(), $group, []);
        if (!is_array($defaults)) {
            return (array) $overrides;
        }

        $merged = $defaults;
        foreach ($overrides as $key => $value) {
            if (is_array($value)) {
                $merged[$key] = $this->normalize($group . '.' . $key, $value, $defaults[$key] ?? []);
            } else {
                $merged[$key] = $this->normalize($group . '.' . $key, $value, $defaults[$key] ?? null);
            }
        }

        return $merged;
    }

    protected function overrides(): array
    {
        if ($this->loaded) {
            return $this->overrides;
        }

        $rows = Setting::query()
            ->where('key', 'like', 'site.%')
            ->get(['key', 'value']);

        foreach ($rows as $row) {
            $key = substr($row->key, 5);
            $decoded = $this->decodeValue($row->value);
            Arr::set($this->overrides, $key, $decoded);
        }

        $this->loaded = true;

        return $this->overrides;
    }

    protected function decodeValue($value)
    {
        if (is_string($value) && $this->isJson($value)) {
            $decoded = json_decode($value, true);
            if (json_last_error() === JSON_ERROR_NONE) {
                return $decoded;
            }
        }
        return $value;
    }

    protected function isJson(string $value): bool
    {
        $value = trim($value);
        if ($value === '') {
            return false;
        }
        return ($value[0] === '{' && str_ends_with($value, '}')) || ($value[0] === '[' && str_ends_with($value, ']'));
    }

    protected function normalize(string $key, $value, $default)
    {
        if (is_array($default)) {
            if (is_string($value)) {
                $lines = preg_split('/\r?\n/', $value);
                $lines = array_filter(array_map('trim', $lines), fn ($line) => $line !== '');
                return array_values($lines ?: $default);
            }
            if (is_array($value)) {
                return $value;
            }
            return $default;
        }

        if (is_string($default)) {
            return is_string($value) ? trim($value) : $default;
        }

        return $value ?? $default;
    }
}


